﻿Texture2D shaderTexture : register(t0);
SamplerState samplerState : register(s0);

cbuffer ConstantBuffer : register(b0)
{
  matrix World : WORLD;
  matrix Projection : PROJECTION;
  matrix ViewProj;
  matrix View : VIEW;
  float4 Parameters;
  float4 MonochromeColor;
};

cbuffer ClipPlaneBuffer : register(b1)
{
  float4 EnableClipping;
  float4 Left;
  float4 Right;
  float4 Top;
  float4 Bottom;
  float4 Front;
  float4 Back;
};

static const float4 grayColor = float4(0.9, 0.9, 0.9, 1);

struct PositionColoredIn
{
  float3 position : POSITION;
  float4 color : COLOR;
};

struct PositionIn
{
  float3 position : POSITION;
};

struct PositionColoredOut
{
  float4 position : SV_POSITION;
  float4 color : COLOR;
  float3 clip0 : SV_CLIPDISTANCE0;
  float3 clip1 : SV_CLIPDISTANCE1;
};

struct PositionTexturedIn
{
  float3 position : POSITION;
  float2 tex : TEXCOORD0;
};

struct PositionTexturedOut
{
  float4 position : SV_POSITION;
  float2 tex : TEXCOORD0;
  float3 clip0 : SV_CLIPDISTANCE0;
  float3 clip1 : SV_CLIPDISTANCE1;
};

void ClipVertex(inout float3 clip0, inout float3 clip1, in float4 worldPos)
{
  if (EnableClipping.x > 0)
  {
    // worldPos.w = 1.0;
    clip0.x = dot(worldPos, Left);
    clip1.x = dot(worldPos, Right);

    clip0.y = dot(worldPos, Top);
    clip1.y = dot(worldPos, Bottom);

    clip0.z = dot(worldPos, Front);
    clip1.z = dot(worldPos, Back);
  }
  else
  {
    clip0 = float3(1, 1, 1);
    clip1 = float3(1, 1, 1);
  }
}

PositionColoredOut VShaderPositionColored(PositionColoredIn vertex)
{
  PositionColoredOut output;

  float4 worldPos = mul(float4(vertex.position, 1), World);

  output.position = worldPos;
  output.position = mul(output.position, ViewProj);

  output.color = Parameters.w > 0 ? grayColor: vertex.color;

  ClipVertex(output.clip0, output.clip1, worldPos);

  return output;
}

PositionColoredOut VShaderPosition(PositionIn vertex)
{
  PositionColoredOut output;

  float4 worldPos = mul(float4(vertex.position, 1), World);

  output.position = worldPos;
  output.position = mul(output.position, ViewProj);

 output.color = Parameters.w > 0 ? grayColor: MonochromeColor;

  ClipVertex(output.clip0, output.clip1, worldPos);

  return output;
}


float4 PShaderPositionColored(PositionColoredOut position) : SV_Target
{
  return position.color;
}

PositionTexturedOut VShaderPositionTextured(PositionTexturedIn vertex)
{
  PositionTexturedOut output;

  float4 worldPos = mul(float4(vertex.position, 1), World);

  output.position = worldPos;
  output.position = mul(output.position, ViewProj);
  output.tex = vertex.tex;

  ClipVertex(output.clip0, output.clip1, worldPos);

  return output;
}

float4 PShaderPositionTextured(PositionTexturedOut position) : SV_Target
{
  if (Parameters.w > 0)
  {
    return grayColor;
  }
  else
  {
    float4 textureColor = Parameters.w > 0 ? grayColor : shaderTexture.Sample(samplerState, position.tex);

    clip(textureColor.a - 0.05);

    return textureColor;
  }
}

PositionColoredIn VShaderPositionColoredPassThrough(PositionColoredIn vertex)
{
  return vertex;
}

PositionColoredIn VShaderPositionPassThrough(PositionIn vertex)
{
  PositionColoredIn result;
  result.position = vertex.position; 
  result.color = Parameters.w > 0 ? grayColor : MonochromeColor;
  return result;
}

[maxvertexcount(8)]
void GS(point PositionColoredIn gin[1], inout TriangleStream<PositionColoredOut> triStream)
{
  float4 worldPos = mul(float4(gin[0].position, 1), World);
  
  float4 position = worldPos;
  position = mul(position, ViewProj);

  if (worldPos.x >= -Left.w && worldPos.y <= Top.w && worldPos.z >= -Front.w)
  {
    float4 v[8];

    float3 up = float3(0.0f, 1.0f, 0.0f);
    float3 right = float3(1.0f, 0.0f, 0.0f);
  	float3 back = float3(0.0f, 0.0f, 1.0f);
    
    v[0] = float4(worldPos.xyz + Parameters.x * right - Parameters.y * up, 1.0f);
    v[1] = float4(worldPos.xyz + Parameters.x * right + Parameters.y * up, 1.0f);
    v[2] = float4(worldPos.xyz - Parameters.x * right - Parameters.y * up, 1.0f);
    v[3] = float4(worldPos.xyz - Parameters.x * right + Parameters.y * up, 1.0f);

  	v[4] = float4(worldPos.xyz + Parameters.z * back - Parameters.y * up, 1.0f);
    v[5] = float4(worldPos.xyz + Parameters.z * back + Parameters.y * up, 1.0f);
    v[6] = float4(worldPos.xyz - Parameters.z * back - Parameters.y * up, 1.0f);
    v[7] = float4(worldPos.xyz - Parameters.z * back + Parameters.y * up, 1.0f);

    PositionColoredOut gout;
    for (int i = 0; i < 4; ++i)
    {
      gout.position = mul(v[i], ViewProj);
      gout.color = Parameters.w > 0 ? grayColor : gin[0].color;
      ClipVertex(gout.clip0,gout.clip1, worldPos);
      triStream.Append(gout);
    }

    triStream.RestartStrip();
  	
  	for (int i = 4; i < 8; ++i)
    {
      gout.position = mul(v[i], ViewProj);
      gout.color = Parameters.w > 0 ? grayColor : gin[0].color;
      ClipVertex(gout.clip0,gout.clip1, worldPos);
      triStream.Append(gout);
    }
  }
}